package control;
/**
 * @author Concetta Balestra, Hani Qusa
 * @version 1.2 09/03/20
 * @since JDK 1.6
 */
import java.util.*;

import abstraction.AvailableServices;
import abstraction.Service;
import abstraction.State;


/**
 * This class permits to create the Cartesian product of all community states and all final states of a set of available services
 *
 */
public class Community {
	
	private AvailableServices availableService;
	
	public Community(AvailableServices as){	
		availableService = as;
	}
	
	/**
	 * gets the service in the specified position
	 * @param index - the position
	 * @return a service
	 */
	public Service getService (int index) {
		return this.availableService.get(index);
	}
	
	
	/**
	 * returns an iterator on community states created on fly
	 * @return Iterator<CState> composed by an array of community state
	 */
	public Iterator<CState> getCommunityStates () {
		return new IteratorCommunityStates(availableService);
	}


	/**
	 * returns on iterator on final community states created on fly
	 * @return Iterator<CState> on final community states
	 */
	public Iterator<CState> getFinalCommunityStates () {
		return new IteratorFinalCommunityStates(availableService);
	}

	/**
	 * gets the initial community state
	 * @return CState
	 */
	public CState getInitialStates () {
		State[] initial  = new State[this.availableService.sizeCommunity()];
		for (int i = 0; i < this.availableService.sizeCommunity(); i++) {
			State s = this.availableService.get(i).getInitialState();
			initial[i]=s;
		}
		return new CState(initial);
	}
	
	/**
	 * gets the position of a service in the set of available services
	 * @param service - the service we want to find
	 * @return an integer
	 */
	public int getServicePosition (Service service) {
		return this.availableService.getPosition(service);
	}
	
	/**
	 * checks if a state of the community is final
	 * @param cs - the community's state to check
	 * @return boolean true if cs is final, false otherwise
	 */
	public boolean isFinalCommunityState(CState cs){
		for(int i=0;i<cs.getSize();i++)
			if (!availableService.get(i).isFinalState(cs.get(i))) return false;
		return true;
	}
	
	/**
	 * community size
	 * @return int
	 */
	public int size(){
		return this.availableService.sizeCommunity();
	}
	
	public String toString () {
		StringBuilder s = new StringBuilder();
		Iterator<CState> it = this.getCommunityStates();
		while (it.hasNext()) {
			CState c = it.next();
			s = s.append("(" + c + ")");
		}
		return s.toString();
	}
	
}

